using System;
using System.Data;
using gov.va.med.vbecs.DAL;
using MethodBase = System.Reflection.MethodBase;
using System.Text.RegularExpressions;
using ADDRESS = gov.va.med.vbecs.Common.VbecsTables.Address;
using EQUIPMENT = gov.va.med.vbecs.Common.VbecsTables.Equipment;


namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carrie Van Stedum</Developer>
	///	<Developer>Luke Meyer</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/17/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Address business object layer class
	///</summary>

	#endregion

	public class Address : BaseBusinessObject 
	{

		#region Variables
		private	const string MIN_COLUMNS = ADDRESS.AddressCity + "," + ADDRESS.AddressGuid + "," + ADDRESS.AddressLine1 + "," + ADDRESS.AddressLine2 + "," + ADDRESS.AddressLine3 + "," + 
											ADDRESS.AddressZipCode + "," + ADDRESS.StateCode;

		/// <summary>
		/// Our Address datatable
		/// </summary>
		private DataTable _dtAddress;

		/// <summary>
		/// The current address data
		/// </summary>
		private DataRow _drAddress;

		private bool _isNew;

		

		#endregion

		#region Constructors

		///<Developers>
		///<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1104"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid address object</ExpectedOutput>
		///</Case>
		///
		///<Case type="1" testid ="203"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Empty address object constructor
		/// </summary>
		public Address()
		{
			this._dtAddress = GetEmptyAddressTableSchema();

			this.MakeNew();

			this._drAddress.AcceptChanges();
			this._dtAddress.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);

		}
		

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2389"> 
		///		<ExpectedInput>Data row containing address data</ExpectedInput>
		///		<ExpectedOutput>Valid Address object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="204"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initiates the new Address object and populates with data from data row
		/// </summary>
		/// <param name="dtRow"></param>
		public Address(DataRow dtRow)
		{
			this._dtAddress = dtRow.Table;
	
			this._drAddress = dtRow;
			
			this.LoadFromDataRow(_drAddress);

			this.IsDirty = false;
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/28/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5054"> 
		///		<ExpectedInput>Valid facilityGuid</ExpectedInput>
		///		<ExpectedOutput>Valid Address object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5055"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initiates the new Address object based on the facility guid
		/// THIS REALLY SHOULD BE A METHOD ON THE COLLECTION FACILITY
		/// </summary>
		/// <param name="facilityGuid"></param>
		public Address(Guid facilityGuid )
		{
			this._dtAddress = GetEmptyAddressTableSchema();

			this._drAddress = _dtAddress.NewRow();
			
			this.LoadFromDataRow( DAL.Address.GetAddress( facilityGuid ) );
			
			this.IsDirty = false;
		}

		#endregion

		#region Events / Event Handlers

		/// <summary>
		/// OnColumnChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="1453"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="211"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update>Unused</Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Loads a data row from an object
		/// </summary>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(ADDRESS.AddressGuid))
			{
				dtRow[ADDRESS.AddressGuid] = this.AddressGuid;
			}

			if(dtRow.Table.Columns.Contains(ADDRESS.AddressLine1))
			{
				if ((this.AddressLine1 != null) && (this.AddressLine1.Trim().Length>0))
				{
					dtRow[ADDRESS.AddressLine1] = this.AddressLine1;
				}
			}
			if(dtRow.Table.Columns.Contains(ADDRESS.AddressLine2))
			{
				if ((this.AddressLine2 != null) && (this.AddressLine2.Trim().Length>0))
				{
					dtRow[ADDRESS.AddressLine2] = this.AddressLine2;
				}
			}
			if(dtRow.Table.Columns.Contains(ADDRESS.AddressLine3))
			{
				if ((this.AddressLine3 != null) && (this.AddressLine3.Trim().Length>0))
				{
					dtRow[ADDRESS.AddressLine3] = this.AddressLine3;
				}
			}
			if(dtRow.Table.Columns.Contains(ADDRESS.AddressCity))
			{
				if ((this.City != null) && (this.City.Trim().Length>0))
				{
					dtRow[ADDRESS.AddressCity] = this.City;
				}
			}
			if(dtRow.Table.Columns.Contains(ADDRESS.StateCode))
			{
				if ((this.StateCode != null) && (this.StateCode.Trim().Length>0))
				{
					dtRow[ADDRESS.StateCode] = this.StateCode;
				}
			}
			if(dtRow.Table.Columns.Contains(ADDRESS.AddressZipCode))
			{
				if ((this.ZipCode != null) && (this.ZipCode.Trim().Length>0))
				{
					dtRow[ADDRESS.AddressZipCode] = this.ZipCode;
				}
			}

			if(dtRow.Table.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion))
			{
				if (this.RowVersion != null)
				{
					dtRow.Table.Columns[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion].ReadOnly = false;
					dtRow[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion] = this.RowVersion;
				}
			}
			return(dtRow);
		}

		
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/26/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2195"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2196"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Creates an empty address table of all our relevant fields...
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptyAddressTableSchema()
		{
			return DAL.Address.GetEmptyAddressTableSchema(true);
		}

		

		/// <summary>
		/// Creates a new blank address datarow
		/// </summary>
		private void MakeNew()
		{
			
			this._drAddress = this._dtAddress.NewRow();

			this._drAddress[ADDRESS.AddressGuid] = System.Guid.NewGuid();
			this._drAddress[ADDRESS.AddressCity] = String.Empty;
			this._drAddress[ADDRESS.AddressZipCode] = String.Empty;
			this._drAddress[ADDRESS.StateCode] = String.Empty;
			this._drAddress[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion] = System.DBNull.Value;

			this._dtAddress.Rows.Add(this._drAddress);
			

			this.IsNew = true;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		/// <summary>
		/// Loads an object from a data row
		/// </summary>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			this.IsNew = false;

			if (dtRow.Table.Columns.Contains(ADDRESS.AddressGuid))
			{
				if (!dtRow.IsNull(ADDRESS.AddressGuid))
				{
					this.AddressGuid = (System.Guid)dtRow[ADDRESS.AddressGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(ADDRESS.AddressLine1))
			{
				if (!dtRow.IsNull(ADDRESS.AddressLine1))
				{
					this.AddressLine1 = dtRow[ADDRESS.AddressLine1].ToString();
				}
				else
				{
					this.AddressLine1 = String.Empty;
				}
			}


			if (dtRow.Table.Columns.Contains(ADDRESS.AddressLine2))
			{
				if (!dtRow.IsNull(ADDRESS.AddressLine2))
				{
					this.AddressLine2 = dtRow[ADDRESS.AddressLine2].ToString();
				}
				else
				{
					this.AddressLine2 = String.Empty;
				}
			}

			if (dtRow.Table.Columns.Contains(ADDRESS.AddressLine3))
			{
				if (!dtRow.IsNull(ADDRESS.AddressLine3))
				{
					this.AddressLine3 = dtRow[ADDRESS.AddressLine3].ToString();
				}
				else
				{
					this.AddressLine3 = String.Empty;
				}
			}

			if (dtRow.Table.Columns.Contains(ADDRESS.AddressCity))
			{
				if (!dtRow.IsNull(ADDRESS.AddressCity))
				{
					this.City = dtRow[ADDRESS.AddressCity].ToString();
				}
				else
				{
					this.City = String.Empty;
				}
			}


			if (dtRow.Table.Columns.Contains(ADDRESS.StateCode))
			{
				if (!dtRow.IsNull(ADDRESS.StateCode))
				{
					this.StateCode = dtRow[ADDRESS.StateCode].ToString();
				}
				else
				{
					this.StateCode = String.Empty;
				}
			}


			if (dtRow.Table.Columns.Contains(ADDRESS.AddressZipCode))
			{
				if (!dtRow.IsNull(ADDRESS.AddressZipCode))
				{
					this.ZipCode = dtRow[ADDRESS.AddressZipCode].ToString();
				}
				else
				{
					this.ZipCode = String.Empty;
				}
			}

			if(dtRow.Table.Columns.Contains(Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion))
			{
				if(!dtRow.IsNull(Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion))
				{
					this.RowVersion = (byte[]) dtRow[Common.DatabaseConstants.ArtificialColumnNames.AddressRowVersion];
				}
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Gregory Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="207"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Address Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="208"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>Unique identifier of the address datarow</summary>
		public System.Guid AddressGuid
		{
			get 
			{
				if (this._drAddress.IsNull(ADDRESS.AddressGuid))
				{
					this._drAddress[ADDRESS.AddressGuid] = System.Guid.NewGuid();
				}
				return (System.Guid)this._drAddress[ADDRESS.AddressGuid];
			}
			set
			{
				if (value == System.Guid.Empty)					
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);

				this._drAddress[ADDRESS.AddressGuid] = value;
				this.IsDirty = true;
			}
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="209"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="210"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Is the object new
		/// </summary>
		public new bool IsNew
		{
			get 
			{ 
				return _isNew;
			}
			set 
			{ 
				_isNew = value;
			}
		}


		///<Developers>
		///	<Developer>Hines CIOFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1105"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1112"> 
		///		<ExpectedInput>Invalid string (256 chars)</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// First line of the address
		/// </summary>
		public string AddressLine1
		{
			get 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.AddressLine1))
				{
					if (this._drAddress.IsNull(ADDRESS.AddressLine1))
					{
						return string.Empty;
					}
					return this._drAddress[ADDRESS.AddressLine1].ToString();
				}
				else
				{
					return string.Empty;
				}
			}
			set 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.AddressLine1))
				{
					this._drAddress[ADDRESS.AddressLine1] = value;
					this.IsDirty = true;
				}
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.FieldLengthError(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											"1", "255").ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1106"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1114"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>BusineesObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Second line of address
		/// </summary>
		public string AddressLine2
		{
			get 
			{
				if (this._drAddress.IsNull(ADDRESS.AddressLine2))
				{
					return string.Empty;
				}
				return this._drAddress[ADDRESS.AddressLine2].ToString();
			}
			set 
			{
				this._drAddress[ADDRESS.AddressLine2] = value;
				this.IsDirty = true;
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.FieldLengthError(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											"1", "255").ResString;
					throw new BusinessObjectException(errorMessage);
				}
			}
		}


		///<Developers>
		///	<Developer>Hines CIOFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1107"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1116"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Third line of address
		/// </summary>
		public string AddressLine3
		{
			get 
			{
				if (this._drAddress.IsNull(ADDRESS.AddressLine3))
				{
					return string.Empty;
				}
				return this._drAddress[ADDRESS.AddressLine3].ToString();
			}
			set 
			{
				this._drAddress[ADDRESS.AddressLine3] = value;
				this.IsDirty = true;
				if (value.Length > 255)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.FieldLengthError(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											"1", "255").ResString;
					throw new BusinessObjectException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>Hines CIOFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1108"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1118"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1119"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Name of the city
		/// </summary>
		public string City
		{
			get 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.AddressCity))
				{
					if (this._drAddress.IsNull(ADDRESS.AddressCity))
					{
						return string.Empty;
					}
					return this._drAddress[ADDRESS.AddressCity].ToString();
				}
				else
				{
					return string.Empty;
				}
			}
			set 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.AddressCity))
				{
					this._drAddress[ADDRESS.AddressCity] = value;
					this.IsDirty = true;
				}
				else
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(this._drAddress).ToString());
				}

				if (value.Length > 55)
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.FieldLengthError(
											Common.Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name, true),
											"1", "55").ResString;
					base.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					throw new BOL.BusinessObjectException(errorMessage);
				}
			
				if (value.Trim().Length == 0)
				{
					base.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}
					
				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
			}
		}
		
		///<Developers>
		///	<Developer>Hines CIOFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1109"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1111"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1120"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// state identifier
		/// </summary>
		public string StateCode
		{
			get 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.StateCode))
				{
					if (this._drAddress.IsNull(ADDRESS.StateCode))
					{
						return string.Empty;
					}
					return this._drAddress[ADDRESS.StateCode].ToString();
				}
				else
				{
					return string.Empty;
				}
			}
			set 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.StateCode))
				{
					this._drAddress[ADDRESS.StateCode] = value;
					this.IsDirty = true;
				}
				else
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(this._drAddress).ToString());
				}

				if (value.Length > 2)
				{
					string lengthError = Common.StrRes.SysErrMsg.Common.StateLength().ResString;
					base.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					throw new BOL.BusinessObjectException(lengthError);
				}

				if (value.Trim().Length == 0)
				{
					base.RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
			}
		}

		///<Developers>
		///	<Developer>Hines CIOFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1110"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No additional broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1121"> 
		///		<ExpectedInput>Invalid string</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="8165"> 
		///		<ExpectedInput>ZipCode column missing from base table</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// city zip code
		/// </summary>
		public string ZipCode
		{
			get 
			{
				if(this._drAddress.Table.Columns.Contains(ADDRESS.AddressZipCode))
				{
					if (this._drAddress.IsNull(ADDRESS.AddressZipCode))
					{
						return string.Empty;
					}
					return this._drAddress[ADDRESS.AddressZipCode].ToString();
				}
				else
				{
					return string.Empty;
				}
			}
			set 
			{
				//Gets 5 or 9 number zip codes
				Regex _regexZipCode = Common.RegularExpressions.ZipCode();
				if(!this._drAddress.Table.Columns.Contains(ADDRESS.AddressZipCode))
				{
					throw new BOL.BusinessObjectException(Common.StrRes.SysErrMsg.Common.InsufficientColumns(this._drAddress).ToString());
				}

				if ((value.Trim().Length > 0) && !_regexZipCode.IsMatch(value.Trim()))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidZipCodeLength().ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}

				this._drAddress[ADDRESS.AddressZipCode] = value;
				this.IsDirty = true;
			}
		}
		

		#endregion
	}
}


